# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

POSCOM Claude v2.0.0 is a deployment framework for Claude Code environments specialized for POS/retail/Odoo development. It provides 77+ specialized AI agents, optimized MCP server configurations, and built-in guardrails.

## Repository Structure

```
poscom-default-claude/
├── agents/               # 77+ specialist agent definitions (.md files)
│   ├── CLAUDE.md         # Guardrails and mandatory rules for all agents
│   ├── chief-agent.md    # Primary orchestrator (start here for complex tasks)
│   └── [specialist].md   # Domain-specific agents
├── bootstrap.sh          # One-line curl installer (requires root)
├── deploy.sh             # Standard deployment script
├── settings.json         # MCP server configuration
└── env.sh                # Environment variables template
```

## Installation & Deployment

```bash
# One-line remote install (as root)
curl -fsSL https://epiccos.iug.net/install-poscom.sh | sudo bash

# Local deployment
./deploy.sh

# After install, configure API keys
nano ~/.claude/poscom-env.sh

# Start Claude with POSCOM agents
claude --agents ~/.claude/agents/poscom
```

## MCP Servers (7 configured)

| Server | Purpose |
|--------|---------|
| memory | Knowledge graph storage |
| thinking | Chain-of-thought reasoning (sequential-thinking) |
| github | Repository operations |
| postgres | Database queries |
| redis | Cache and sessions |
| fetch | Web fetching with image support |
| playwright | Browser automation |

## Agent Architecture

### Hierarchy
The Chief Agent (`chief-agent.md`) is the primary orchestrator that:
- Analyzes tasks and assembles specialist teams
- Creates new agents dynamically when needed
- Launches parallel workstreams
- Validates all outputs before completion
- Never stops until the ultimate goal is achieved

### Agent Categories (77+ agents)
- **Core Orchestration**: chief-agent, architect
- **Development**: backend-developer, frontend-developer, fullstack-developer
- **Quality**: test-automator, security-auditor, code-reviewer
- **Data**: database-architect, postgresql-expert, redis-expert
- **Odoo**: odoo-developer, odoo-pos-specialist, odoo-workflow-architect + 6 more
- **Domain**: payment-integration-expert, inventory-specialist, loyalty-program-expert
- **Marketing**: seo-llm-marketing-expert, marketing-strategist
- **Enterprise**: o365-teams-automation-expert

### When to Use Chief Agent
Always invoke chief-agent first for:
- Multi-step tasks
- Tasks requiring multiple skills (backend + frontend + testing)
- Complex features or implementations
- Project planning and architecture decisions

## Mandatory Rules (All Agents)

### Before Code Changes
1. Understand full context - ask questions if unclear
2. Check existing tests pass before making changes
3. Use TodoWrite to track multi-step work

### During Development
- Never commit secrets or credentials
- Always use parameterized queries (no SQL injection)
- Always validate user input
- Write tests for new functionality
- Follow existing code patterns

### Before Marking Complete
- All tests pass
- No security vulnerabilities (high/critical)
- Code reviewed for changes > 20 lines
- Documentation updated if public API changed

## Blocked Operations

```
- rm -rf /
- Storing credit card numbers
- Disabling SSL/TLS verification
- Committing .env files
- Hardcoding API keys
- eval() or exec() with user input
```

## POS-Specific Standards

### Payment Processing
- Use payment processor tokenization
- Never store CVV/CVC
- Log transaction IDs, not card data
- Implement idempotency keys

### Customer Data
- Encrypt PII at rest
- Mask sensitive data in logs
- GDPR/CCPA compliance

### Inventory
- Use optimistic locking
- Audit trail for all changes
- Real-time sync with POS terminals

## Quality Gates

| Gate | Requirement | Enforcer |
|------|-------------|----------|
| Unit Tests | 100% pass | test-automator |
| Integration | 100% pass | test-automator |
| Security | No high/critical | security-auditor |
| Coverage | >= 80% on critical paths | test-automator |

## Odoo Spec Workflow

For Odoo module development, use the claude-code-spec-workflow-odoo framework:

```bash
npm i -g @stanleykao72/claude-code-spec-workflow-odoo
claude-code-spec-workflow-odoo setup
claude-code-spec-workflow-odoo odoo-setup
```

Key commands:
- `/odoo-spec-create <name> "desc"` - Create specification
- `/odoo-spec-execute <task-id> <spec>` - Execute task
- `/odoo-bug-create <name> "desc"` - Document bug

## Environment Variables

Required in `~/.claude/poscom-env.sh`:
```bash
export ANTHROPIC_API_KEY="sk-ant-..."
export GITHUB_TOKEN="ghp_..."
export POSTGRES_CONNECTION="postgresql://user:pass@host/db"
export REDIS_CONNECTION="redis://localhost:6379"
```

## Creating New Agents

Add a new agent by creating `agents/{agent-name}.md`:

```markdown
---
name: agent-name
description: What this agent does
tools: Read, Write, Edit, Bash, Grep, Glob
---

You are a specialist in...

## Expertise Areas
...

## Patterns and Standards
...

## Quality Checklist
...
```
