---
name: odoo-dev-reference
description: Comprehensive Odoo 18/19 Development Reference - API changes, deprecated fields, standards, and migration guide
version: "2.0"
odoo_versions:
  - "14.0"
  - "15.0"
  - "16.0"
  - "17.0"
  - "18.0"
  - "19.0"
---

# Odoo Development Reference v2.0

> **CRITICAL**: All Odoo agents MUST reference this document for version-specific development.
> This document is the authoritative source for deprecated fields, API changes, and coding standards.

## Table of Contents

1. [Odoo 18 Changes](#odoo-18-changes)
2. [Odoo 19 Requirements](#odoo-19-requirements-preview)
3. [Deprecated Fields & Methods](#deprecated-fields--methods)
4. [Field Types Reference](#field-types-reference)
5. [ORM API Reference](#orm-api-reference)
6. [Coding Standards](#coding-standards)
7. [Migration Patterns](#migration-patterns)
8. [Testing Standards](#testing-standards)
9. [Security Patterns](#security-patterns)
10. [Performance Guidelines](#performance-guidelines)

---

## Odoo 18 Changes

### New Features in Odoo 18

#### 1. Aggregator Field Attribute (BREAKING CHANGE)

```python
# OLD (Deprecated in 18.0) - DO NOT USE
class MyModel(models.Model):
    amount = fields.Float(group_operator='sum')  # DEPRECATED

# NEW (Odoo 18+) - USE THIS
class MyModel(models.Model):
    amount = fields.Float(aggregator='sum')  # CORRECT
```

**Supported aggregators:**
- `sum` - Sum of all values
- `avg` - Arithmetic mean
- `max` - Maximum value
- `min` - Minimum value
- `count` - Number of rows
- `count_distinct` - Number of distinct rows
- `bool_and` - True if all values are true
- `bool_or` - True if at least one value is true
- `array_agg` - Values concatenated into array
- `recordset` - Values as recordset (post-processed)

#### 2. Enhanced SQL Safety

```python
# Odoo 18 introduces SQL class for safer queries
from odoo.tools import SQL

# OLD (Vulnerable to SQL injection)
self.env.cr.execute("SELECT * FROM %s WHERE id = %s" % (table, id))

# NEW (Safe SQL construction)
self.env.cr.execute(SQL(
    "SELECT * FROM %s WHERE id = %s",
    SQL.identifier(table),
    id
))

# SQL identifier for table/column names
SQL.identifier('my_table')  # Properly quoted identifier
SQL.identifier('my_table', 'my_column')  # table.column
```

#### 3. Company-Dependent Fields (Enhanced)

```python
# Odoo 18 stores company-dependent fields as JSONB
class MyModel(models.Model):
    # Stored as jsonb with company_id as key
    property_cost = fields.Float(company_dependent=True)

    # Allowed types for company_dependent:
    # char, float, boolean, integer, text, many2one, date, datetime, selection, html

# Company-dependent fields automatically get:
# - index='btree_not_null' (for performance)
# - prefetch='company_dependent' (group prefetching)
# - copy=False (not copied by default)
# - _depends_context = ('company',)
```

#### 4. Precompute Attribute

```python
class MyModel(models.Model):
    # Compute field value BEFORE database insert
    sequence = fields.Integer(compute='_compute_sequence', store=True, precompute=True)

    @api.depends('parent_id')
    def _compute_sequence(self):
        for record in self:
            record.sequence = (record.parent_id.sequence or 0) + 10

# WARNING: precompute only works when:
# - No explicit value provided to create()
# - No default value defined
# - Records created in batch (otherwise less efficient than flush)
```

#### 5. Allow Sudo Commands (Security)

```python
class SensitiveModel(models.Model):
    _name = 'sensitive.model'

    # Prevent manipulation through One2many/Many2many in sudo mode
    _allow_sudo_commands = False  # Default is True

    # This protects against:
    # - Malicious command manipulation
    # - Privilege escalation through relational fields
```

#### 6. Export String Translation

```python
class MyModel(models.Model):
    name = fields.Char(
        string='Name',
        export_string_translation=True,  # Export label translations (default)
    )

    internal_code = fields.Char(
        string='Internal Code',
        export_string_translation=False,  # Don't export translations
    )
```

#### 7. Enhanced Index Types

```python
class MyModel(models.Model):
    # Index options in Odoo 18:
    name = fields.Char(index='btree')          # Standard BTREE (or index=True)
    code = fields.Char(index='btree_not_null') # BTREE without NULL values
    description = fields.Text(index='trigram') # GIN with trigrams (full-text)
    no_index = fields.Char(index=False)        # No index (default for most)
```

#### 8. Read Group Enhancements

```python
# New granularity options for date/datetime grouping
READ_GROUP_TIME_GRANULARITY = {
    'hour': relativedelta(hours=1),
    'day': relativedelta(days=1),
    'week': timedelta(days=7),
    'month': relativedelta(months=1),
    'quarter': relativedelta(months=3),
    'year': relativedelta(years=1)
}

READ_GROUP_NUMBER_GRANULARITY = {
    'year_number': 'year',
    'quarter_number': 'quarter',
    'month_number': 'month',
    'iso_week_number': 'week',
    'day_of_year': 'doy',
    'day_of_month': 'day',
    'day_of_week': 'dow',
    'hour_number': 'hour',
    'minute_number': 'minute',
    'second_number': 'second',
}

# Usage
results = self.env['sale.order'].read_group(
    domain=[],
    groupby=['date_order:month'],
    aggregates=['amount_total:sum']
)
```

#### 9. Properties Field (Dynamic Fields)

```python
class MyModel(models.Model):
    # Properties field for dynamic key-value pairs
    properties = fields.Properties(
        definition='definition_id.properties_definition',
    )

    # Properties can have types:
    # - char, integer, float, boolean
    # - date, datetime
    # - many2one, many2many
    # - selection, tags
```

#### 10. Check Company Auto

```python
class MyModel(models.Model):
    _name = 'my.model'
    _check_company_auto = True  # Auto-validate company on write/create

    company_id = fields.Many2one('res.company')
    partner_id = fields.Many2one(
        'res.partner',
        check_company=True,  # Validated if _check_company_auto=True
    )
```

---

## Odoo 19 Requirements (Preview)

> **Note**: Odoo 19 is expected to release in late 2025. These are anticipated requirements based on Odoo's development trajectory.

### Expected Changes in Odoo 19

#### 1. Full Python 3.12+ Requirement

```python
# Odoo 19 expected minimum: Python 3.12
# Key Python 3.12 features to prepare for:

# Type parameter syntax (PEP 695)
type Point = tuple[float, float]

# Per-interpreter GIL (PEP 684) - improved multiprocessing
# Improved error messages
# F-string improvements
```

#### 2. OWL 3.0 Frontend Framework

```javascript
// Expected OWL 3.0 changes:
// - New reactivity system
// - Enhanced hooks API
// - Improved TypeScript support
// - Better async component handling

import { Component, useState, onMounted } from "@odoo/owl";

class MyComponent extends Component {
    static template = "my_module.MyComponent";

    setup() {
        this.state = useState({
            count: 0,
        });

        onMounted(() => {
            // Component mounted
        });
    }
}
```

#### 3. Enhanced API Deprecations

```python
# Expected removals in Odoo 19:
# - Final removal of group_operator (use aggregator)
# - Removal of legacy JS widgets
# - Removal of deprecated ORM methods
# - Stricter type checking

# Prepare by:
# 1. Removing all group_operator usage
# 2. Migrating to OWL components
# 3. Using modern API patterns
# 4. Adding type hints
```

#### 4. Async ORM Operations

```python
# Expected async ORM support in Odoo 19
# Prepare modules for async patterns:

class MyModel(models.Model):
    # Current synchronous pattern
    def process_records(self):
        for record in self:
            record._process_single()

    # Prepare for async (structure code for easy migration)
    def _process_single(self):
        # Isolated processing logic
        pass
```

#### 5. Module Manifest Changes

```python
# Expected __manifest__.py changes for Odoo 19:
{
    'name': 'My Module',
    'version': '19.0.1.0.0',
    'category': 'Sales',
    'summary': 'Module summary',
    'description': """Long description""",
    'depends': ['base', 'sale'],
    'data': [],
    'assets': {
        'web.assets_backend': [
            'my_module/static/src/**/*',
        ],
    },
    # Expected new keys:
    'python_requires': '>=3.12',
    'odoo_requires': '>=19.0',
    'license': 'LGPL-3',  # Will become required
    'maintainer': 'Company Name',
}
```

---

## Deprecated Fields & Methods

### Deprecated in Odoo 18

| Deprecated | Replacement | Version | Notes |
|------------|-------------|---------|-------|
| `group_operator` | `aggregator` | 18.0 | Field attribute for read_group |
| `_constraints` | `@api.constrains` | 16.0 (removed 18) | Python constraints |
| `multi=True` on fields | N/A | 17.0 | No longer needed |
| `states` on fields | Python logic | 17.0 | Use `readonly`/`invisible` with logic |
| `track_visibility` | `tracking=True` | 15.0 | Simplified tracking |

### Deprecated in Odoo 17

| Deprecated | Replacement | Notes |
|------------|-------------|-------|
| `oldname` on fields | Data migration script | Field renaming |
| `required_if_provider` | `@api.constrains` | Payment provider pattern |
| Legacy JS widgets | OWL components | Complete migration |
| `kanban_view_ref` | `action` context | Kanban selection |

### Deprecated in Odoo 16

| Deprecated | Replacement | Notes |
|------------|-------------|-------|
| `@api.one` | `@api.depends` loop | Removed entirely |
| `@api.multi` | Default behavior | Removed entirely |
| `api.v7` patterns | Modern API | Removed entirely |
| `openerp` imports | `odoo` imports | Removed entirely |

### Removed Patterns (DO NOT USE)

```python
# WRONG - These will FAIL in Odoo 18+:

# 1. Old constraint syntax
_constraints = [
    ('check_amount', 'Amount must be positive'),
]

# 2. group_operator
amount = fields.Float(group_operator='sum')

# 3. oldname for field rename
new_field = fields.Char(oldname='old_field')

# 4. api.one decorator
@api.one
def compute_something(self):
    self.field = 'value'

# 5. openerp imports
from openerp import models, fields  # WRONG
```

---

## Field Types Reference

### Core Field Types (Odoo 18)

```python
from odoo import fields, models

class FieldReference(models.Model):
    _name = 'field.reference'

    # Basic Types
    char_field = fields.Char(string='String', size=256, trim=True)
    text_field = fields.Text(string='Long Text')
    html_field = fields.Html(string='HTML Content', sanitize=True)
    integer_field = fields.Integer(string='Integer', aggregator='sum')
    float_field = fields.Float(string='Float', digits=(16, 2), aggregator='avg')
    monetary_field = fields.Monetary(string='Amount', currency_field='currency_id')
    boolean_field = fields.Boolean(string='Active', default=True)

    # Date/Time Types
    date_field = fields.Date(string='Date', default=fields.Date.today)
    datetime_field = fields.Datetime(string='DateTime', default=fields.Datetime.now)

    # Selection
    state = fields.Selection([
        ('draft', 'Draft'),
        ('confirmed', 'Confirmed'),
        ('done', 'Done'),
    ], string='State', default='draft', required=True)

    # Binary
    image = fields.Image(string='Image', max_width=1024, max_height=1024)
    file = fields.Binary(string='File', attachment=True)

    # Relational
    partner_id = fields.Many2one('res.partner', string='Partner', ondelete='cascade')
    tag_ids = fields.Many2many('my.tag', string='Tags')
    line_ids = fields.One2many('my.line', 'parent_id', string='Lines')

    # Reference (dynamic model)
    reference = fields.Reference(
        selection=[('res.partner', 'Partner'), ('res.users', 'User')],
        string='Reference',
    )

    # Computed
    computed_field = fields.Char(
        string='Computed',
        compute='_compute_field',
        store=True,
        precompute=True,
    )

    # Company Dependent (Property)
    property_field = fields.Float(
        string='Property',
        company_dependent=True,
    )

    # Related
    partner_name = fields.Char(
        related='partner_id.name',
        string='Partner Name',
        store=True,
    )

    # Properties (Dynamic)
    properties = fields.Properties(
        definition='category_id.properties_definition',
    )
```

### Field Attributes Quick Reference

| Attribute | Type | Default | Description |
|-----------|------|---------|-------------|
| `string` | str | Field name | Label for UI |
| `help` | str | None | Tooltip text |
| `readonly` | bool | False | Read-only in UI |
| `required` | bool | False | Mandatory field |
| `index` | str/bool | False | Database index type |
| `default` | value/callable | None | Default value |
| `groups` | str | None | Access groups (CSV) |
| `company_dependent` | bool | False | Per-company values |
| `copy` | bool | True | Copy on duplicate |
| `store` | bool | True | Store in database |
| `compute` | str | None | Compute method name |
| `inverse` | str | None | Inverse method name |
| `search` | str | None | Search method name |
| `related` | str | None | Related field path |
| `aggregator` | str | None | Aggregation function |
| `precompute` | bool | False | Compute before insert |
| `tracking` | bool/int | False | Track changes |
| `translate` | bool/callable | False | Translatable |

---

## ORM API Reference

### CRUD Operations

```python
# CREATE
record = self.env['model.name'].create({
    'name': 'Value',
    'field': value,
})

# CREATE with batch (more efficient)
records = self.env['model.name'].create([
    {'name': 'Value 1'},
    {'name': 'Value 2'},
])

# READ
record = self.env['model.name'].browse(id)
records = self.env['model.name'].search([('field', '=', value)])
data = records.read(['field1', 'field2'])

# UPDATE
record.write({'field': new_value})

# DELETE
record.unlink()
```

### Search Domain Operators

```python
# Comparison operators
[('field', '=', value)]      # Equal
[('field', '!=', value)]     # Not equal
[('field', '>', value)]      # Greater than
[('field', '>=', value)]     # Greater or equal
[('field', '<', value)]      # Less than
[('field', '<=', value)]     # Less or equal

# String operators
[('field', 'like', 'pattern')]     # SQL LIKE (case sensitive)
[('field', 'ilike', 'pattern')]    # SQL ILIKE (case insensitive)
[('field', '=like', 'pattern')]    # SQL = with pattern
[('field', '=ilike', 'pattern')]   # Case insensitive =like

# List operators
[('field', 'in', [1, 2, 3])]       # In list
[('field', 'not in', [1, 2, 3])]   # Not in list

# Relational operators
[('child_ids', 'child_of', id)]    # Hierarchical child
[('parent_id', 'parent_of', id)]   # Hierarchical parent

# Boolean operators
'&'  # AND (default)
'|'  # OR
'!'  # NOT

# Complex domain
['|',
    ('state', '=', 'done'),
    '&',
        ('state', '=', 'draft'),
        ('user_id', '=', uid)
]
```

### Decorators Reference

```python
from odoo import api, models

class MyModel(models.Model):

    @api.depends('field1', 'field2')
    def _compute_result(self):
        """Compute method - triggered when dependencies change."""
        for record in self:
            record.result = record.field1 + record.field2

    @api.depends_context('company')
    def _compute_company_result(self):
        """Compute based on context (e.g., company)."""
        for record in self:
            record.result = self.env.company.name

    @api.constrains('field1', 'field2')
    def _check_consistency(self):
        """Validation constraint - raises ValidationError."""
        for record in self:
            if record.field1 > record.field2:
                raise ValidationError("Field1 must be <= Field2")

    @api.onchange('field1')
    def _onchange_field1(self):
        """UI onchange - sets values and returns warnings."""
        if self.field1 > 100:
            self.field2 = self.field1 * 2
            return {
                'warning': {
                    'title': 'Warning',
                    'message': 'Field1 is high!',
                }
            }

    @api.ondelete(at_uninstall=False)
    def _check_delete(self):
        """Check before delete - raises UserError."""
        for record in self:
            if record.state == 'done':
                raise UserError("Cannot delete done records")

    @api.model
    def some_method(self):
        """Class-level method (self is empty recordset)."""
        return self.env['res.partner'].search([])

    @api.model_create_multi
    def create(self, vals_list):
        """Override create for batch processing."""
        records = super().create(vals_list)
        for record in records:
            record._post_create()
        return records

    @api.returns('self', lambda value: value.id)
    def copy(self, default=None):
        """Custom copy with id return."""
        return super().copy(default)

    @api.autovacuum
    def _gc_cleanup(self):
        """Garbage collection method for transient models."""
        self.search([('create_date', '<', limit_date)]).unlink()
```

---

## Coding Standards

### Python Standards for Odoo 18/19

```python
# 1. Type hints (encouraged for Odoo 19)
from odoo import api, fields, models
from typing import Optional, List, Dict, Any

class MyModel(models.Model):
    _name = 'my.model'
    _description = 'My Model'

    def process_data(self, data: Dict[str, Any]) -> List[int]:
        """Process data and return record IDs.

        Args:
            data: Dictionary with processing parameters

        Returns:
            List of created record IDs
        """
        result: List[int] = []
        for item in data.get('items', []):
            record = self.create(item)
            result.append(record.id)
        return result

# 2. Docstrings (required)
def my_method(self):
    """Short description of method purpose.

    Longer description if needed, explaining the business
    logic and any important considerations.

    Returns:
        Description of return value

    Raises:
        UserError: When validation fails
        AccessError: When user lacks permissions
    """
    pass

# 3. Private methods (underscore prefix)
def _compute_field(self):
    """Private compute method."""
    pass

def _helper_method(self):
    """Private helper - not exposed to external calls."""
    pass

# 4. Exception handling
from odoo.exceptions import UserError, ValidationError, AccessError

try:
    result = risky_operation()
except ValueError as e:
    raise UserError(f"Invalid value: {e}") from e
except AccessError:
    raise  # Re-raise access errors
except Exception:
    _logger.exception("Unexpected error in operation")
    raise UserError("An unexpected error occurred")
```

### Module Structure

```
my_module/
├── __init__.py
├── __manifest__.py
├── controllers/
│   ├── __init__.py
│   └── main.py
├── data/
│   ├── data.xml
│   └── demo.xml
├── i18n/
│   ├── my_module.pot
│   └── fr.po
├── models/
│   ├── __init__.py
│   ├── my_model.py
│   └── inherited_model.py
├── report/
│   ├── __init__.py
│   └── report_template.xml
├── security/
│   ├── ir.model.access.csv
│   └── security.xml
├── static/
│   ├── description/
│   │   ├── icon.png
│   │   └── index.html
│   └── src/
│       ├── js/
│       ├── scss/
│       └── xml/
├── tests/
│   ├── __init__.py
│   └── test_my_model.py
├── views/
│   ├── my_model_views.xml
│   ├── menus.xml
│   └── templates.xml
└── wizard/
    ├── __init__.py
    └── my_wizard.py
```

### XML Standards

```xml
<?xml version="1.0" encoding="utf-8"?>
<odoo>
    <!-- Record XML ID format: module_name.model_action_type -->

    <!-- Form View -->
    <record id="my_module_my_model_view_form" model="ir.ui.view">
        <field name="name">my.model.form</field>
        <field name="model">my.model</field>
        <field name="arch" type="xml">
            <form string="My Model">
                <header>
                    <button name="action_confirm" type="object"
                            string="Confirm" class="btn-primary"
                            invisible="state != 'draft'"/>
                    <field name="state" widget="statusbar"/>
                </header>
                <sheet>
                    <div class="oe_title">
                        <h1>
                            <field name="name" placeholder="Name..."/>
                        </h1>
                    </div>
                    <group>
                        <group>
                            <field name="partner_id"/>
                            <field name="date"/>
                        </group>
                        <group>
                            <field name="amount"/>
                            <field name="currency_id"/>
                        </group>
                    </group>
                    <notebook>
                        <page string="Lines" name="lines">
                            <field name="line_ids">
                                <list editable="bottom">
                                    <field name="product_id"/>
                                    <field name="quantity"/>
                                    <field name="price_unit"/>
                                </list>
                            </field>
                        </page>
                        <page string="Notes" name="notes">
                            <field name="notes"/>
                        </page>
                    </notebook>
                </sheet>
                <chatter/>
            </form>
        </field>
    </record>

    <!-- List View -->
    <record id="my_module_my_model_view_list" model="ir.ui.view">
        <field name="name">my.model.list</field>
        <field name="model">my.model</field>
        <field name="arch" type="xml">
            <list string="My Models" multi_edit="1">
                <field name="name"/>
                <field name="partner_id"/>
                <field name="date"/>
                <field name="amount" sum="Total"/>
                <field name="state" widget="badge"
                       decoration-info="state == 'draft'"
                       decoration-success="state == 'done'"/>
            </list>
        </field>
    </record>

    <!-- Search View -->
    <record id="my_module_my_model_view_search" model="ir.ui.view">
        <field name="name">my.model.search</field>
        <field name="model">my.model</field>
        <field name="arch" type="xml">
            <search string="Search Models">
                <field name="name"/>
                <field name="partner_id"/>
                <filter string="Draft" name="draft"
                        domain="[('state', '=', 'draft')]"/>
                <filter string="Confirmed" name="confirmed"
                        domain="[('state', '=', 'confirmed')]"/>
                <separator/>
                <filter string="This Month" name="this_month"
                        domain="[('date', '>=', (context_today() - relativedelta(day=1)).strftime('%Y-%m-%d'))]"/>
                <group expand="0" string="Group By">
                    <filter string="Partner" name="partner"
                            context="{'group_by': 'partner_id'}"/>
                    <filter string="State" name="state"
                            context="{'group_by': 'state'}"/>
                    <filter string="Month" name="month"
                            context="{'group_by': 'date:month'}"/>
                </group>
            </search>
        </field>
    </record>

    <!-- Action -->
    <record id="my_module_my_model_action" model="ir.actions.act_window">
        <field name="name">My Models</field>
        <field name="res_model">my.model</field>
        <field name="view_mode">list,form,kanban</field>
        <field name="context">{'search_default_draft': 1}</field>
        <field name="help" type="html">
            <p class="o_view_nocontent_smiling_face">
                Create your first record
            </p>
        </field>
    </record>

    <!-- Menu -->
    <menuitem id="my_module_menu_root"
              name="My Module"
              sequence="10"/>
    <menuitem id="my_module_menu_models"
              name="Models"
              parent="my_module_menu_root"
              action="my_module_my_model_action"
              sequence="10"/>
</odoo>
```

---

## Migration Patterns

### Version Upgrade Checklist

#### From 17.0 to 18.0

```python
# 1. Replace group_operator with aggregator
# BEFORE
amount = fields.Float(group_operator='sum')
# AFTER
amount = fields.Float(aggregator='sum')

# 2. Update SQL queries to use SQL class
# BEFORE
self.env.cr.execute("SELECT * FROM %s" % table)
# AFTER
from odoo.tools import SQL
self.env.cr.execute(SQL("SELECT * FROM %s", SQL.identifier(table)))

# 3. Review company_dependent fields (now JSONB)
# Ensure proper migration of property fields

# 4. Update OWL components to 2.0 patterns
# Check frontend JS for deprecated widget patterns

# 5. Review _allow_sudo_commands for sensitive models
```

#### From 16.0 to 17.0

```python
# 1. Migrate kanban_view_ref to action context
# 2. Update legacy JS to OWL components
# 3. Remove states attribute, use Python logic
# 4. Update deprecated payment provider patterns
```

### Data Migration Script Template

```python
# migrations/18.0.1.0.0/pre-migration.py
from openupgradelib import openupgrade

@openupgrade.migrate()
def migrate(env, version):
    """Pre-migration: prepare database before module update."""
    openupgrade.logged_query(
        env.cr,
        """
        ALTER TABLE my_model
        ADD COLUMN IF NOT EXISTS new_field VARCHAR
        """,
    )

# migrations/18.0.1.0.0/post-migration.py
from openupgradelib import openupgrade

@openupgrade.migrate()
def migrate(env, version):
    """Post-migration: data transformations after update."""
    # Migrate data from old field to new field
    openupgrade.logged_query(
        env.cr,
        """
        UPDATE my_model
        SET new_field = old_field
        WHERE old_field IS NOT NULL
        """,
    )

    # Use ORM for complex migrations
    records = env['my.model'].search([('state', '=', 'old_state')])
    records.write({'state': 'new_state'})
```

---

## Testing Standards

### pytest-odoo Test Structure

```python
# tests/test_my_model.py
from odoo.tests import TransactionCase, tagged
from odoo.exceptions import UserError, ValidationError

@tagged('post_install', '-at_install')
class TestMyModel(TransactionCase):

    @classmethod
    def setUpClass(cls):
        """Set up test fixtures."""
        super().setUpClass()
        cls.partner = cls.env['res.partner'].create({
            'name': 'Test Partner',
        })
        cls.model = cls.env['my.model']

    def test_create_basic(self):
        """Test basic record creation."""
        record = self.model.create({
            'name': 'Test',
            'partner_id': self.partner.id,
        })
        self.assertEqual(record.name, 'Test')
        self.assertEqual(record.partner_id, self.partner)

    def test_constraint_validation(self):
        """Test validation constraints raise errors."""
        with self.assertRaises(ValidationError):
            self.model.create({
                'name': '',  # Empty name should fail
            })

    def test_compute_field(self):
        """Test computed field calculation."""
        record = self.model.create({
            'name': 'Test',
            'field1': 10,
            'field2': 20,
        })
        self.assertEqual(record.computed_total, 30)

    def test_workflow_transition(self):
        """Test state machine transitions."""
        record = self.model.create({'name': 'Test'})
        self.assertEqual(record.state, 'draft')

        record.action_confirm()
        self.assertEqual(record.state, 'confirmed')

        record.action_done()
        self.assertEqual(record.state, 'done')

    def test_access_rights(self):
        """Test security and access control."""
        user = self.env['res.users'].create({
            'name': 'Test User',
            'login': 'testuser',
            'groups_id': [(6, 0, [self.env.ref('base.group_user').id])],
        })

        record = self.model.sudo(user).create({'name': 'Test'})
        self.assertTrue(record.exists())


@tagged('post_install', '-at_install')
class TestMyModelIntegration(TransactionCase):
    """Integration tests for cross-model operations."""

    def test_full_workflow(self):
        """Test complete business workflow."""
        # Create order
        order = self.env['sale.order'].create({...})

        # Add lines
        order.write({'order_line': [(0, 0, {...})]})

        # Confirm
        order.action_confirm()

        # Validate inventory
        picking = order.picking_ids[0]
        picking.action_confirm()
        picking.action_done()

        # Create invoice
        invoice = order._create_invoices()
        invoice.action_post()

        # Assert final state
        self.assertEqual(order.state, 'sale')
        self.assertEqual(order.invoice_status, 'invoiced')
```

### Running Tests

```bash
# Run all tests for a module
odoo-bin -c odoo.conf -d testdb --test-enable -i my_module --stop-after-init

# Run specific test class
odoo-bin -c odoo.conf -d testdb --test-tags=my_model

# Run with pytest-odoo
pytest --odoo-database=testdb addons/my_module/tests/

# Run with coverage
pytest --odoo-database=testdb --cov=addons/my_module addons/my_module/tests/
```

---

## Security Patterns

### Access Control

```xml
<!-- security/ir.model.access.csv -->
id,name,model_id:id,group_id:id,perm_read,perm_write,perm_create,perm_unlink
access_my_model_user,my.model.user,model_my_model,base.group_user,1,0,0,0
access_my_model_manager,my.model.manager,model_my_model,my_module.group_manager,1,1,1,1

<!-- security/security.xml -->
<odoo>
    <!-- Groups -->
    <record id="group_manager" model="res.groups">
        <field name="name">My Module Manager</field>
        <field name="implied_ids" eval="[(4, ref('base.group_user'))]"/>
    </record>

    <!-- Record Rules -->
    <record id="rule_my_model_user" model="ir.rule">
        <field name="name">My Model: User sees own records</field>
        <field name="model_id" ref="model_my_model"/>
        <field name="domain_force">[('user_id', '=', user.id)]</field>
        <field name="groups" eval="[(4, ref('base.group_user'))]"/>
    </record>

    <record id="rule_my_model_company" model="ir.rule">
        <field name="name">My Model: Multi-company</field>
        <field name="model_id" ref="model_my_model"/>
        <field name="domain_force">
            ['|', ('company_id', '=', False), ('company_id', 'in', company_ids)]
        </field>
    </record>
</odoo>
```

### Field-Level Security

```python
class MyModel(models.Model):
    _name = 'my.model'

    # Visible only to managers
    secret_field = fields.Char(
        groups='my_module.group_manager',
    )

    # Invisible in ORM (security through obscurity - use carefully)
    internal_field = fields.Char(
        groups='.',  # No access
    )

    # Check access in methods
    def action_sensitive(self):
        if not self.env.user.has_group('my_module.group_manager'):
            raise AccessError("You don't have access to this action")
        # Perform action
```

---

## Performance Guidelines

### Query Optimization

```python
# 1. Avoid N+1 queries
# BAD
for record in records:
    partner = record.partner_id  # Query per record

# GOOD
records = records.with_prefetch(records.ids)
for record in records:
    partner = record.partner_id  # Prefetched

# 2. Use read() for large datasets
# BAD
data = [(r.name, r.amount) for r in records]  # Multiple queries

# GOOD
data = records.read(['name', 'amount'])  # Single query

# 3. Use search_read() when possible
results = self.env['my.model'].search_read(
    domain=[('state', '=', 'done')],
    fields=['name', 'amount'],
    limit=100,
)

# 4. Use SQL for heavy operations
from odoo.tools import SQL

self.env.cr.execute(SQL("""
    SELECT partner_id, SUM(amount)
    FROM my_model
    WHERE state = 'done'
    GROUP BY partner_id
"""))
results = self.env.cr.fetchall()

# 5. Index frequently searched fields
name = fields.Char(index='btree')
date = fields.Date(index='btree')
state = fields.Selection([...], index=True)
```

### Caching

```python
from odoo.tools import ormcache

class MyModel(models.Model):
    _name = 'my.model'

    @ormcache('self.env.uid', 'category_id')
    def _get_category_rules(self, category_id):
        """Cached method for expensive computations."""
        return self.env['my.rule'].search([
            ('category_id', '=', category_id),
        ]).ids

    def invalidate_cache(self):
        """Clear cache when rules change."""
        self._get_category_rules.clear_cache(self)
```

### Batch Processing

```python
# Process in batches to avoid memory issues
BATCH_SIZE = 1000

records = self.env['my.model'].search([('state', '=', 'pending')])
for batch in split_every(BATCH_SIZE, records.ids):
    batch_records = self.env['my.model'].browse(batch)
    batch_records._process_batch()
    self.env.cr.commit()  # Commit each batch
```

---

## Version Compatibility Table

| Feature | 14.0 | 15.0 | 16.0 | 17.0 | 18.0 | 19.0 |
|---------|------|------|------|------|------|------|
| Python 3.8+ | Required | Required | Required | Required | - | - |
| Python 3.10+ | - | - | Required | Required | Required | - |
| Python 3.12+ | - | - | - | - | - | Required |
| OWL 1.x | - | Intro | Stable | - | - | - |
| OWL 2.x | - | - | - | Stable | Stable | - |
| OWL 3.x | - | - | - | - | - | Expected |
| PostgreSQL 12+ | Required | Required | Required | Required | Required | - |
| PostgreSQL 14+ | - | - | - | Required | Required | Required |
| `aggregator` | - | - | - | - | Required | Required |
| `group_operator` | OK | OK | OK | Deprecated | Deprecated | Removed |
| `@api.one` | Deprecated | Removed | Removed | Removed | Removed | Removed |
| JSONB Properties | - | - | - | Intro | Stable | Stable |
| Assets Bundling | Legacy | New | Stable | Stable | Stable | Stable |

---

## Quick Reference Card

### Model Attributes

```python
_name = 'module.model'          # Technical name
_description = 'Human Name'     # UI description (required)
_inherit = ['parent.model']     # Inherit from
_inherits = {'parent': 'x_id'}  # Delegation
_table = 'custom_table'         # Custom table name
_order = 'date desc, name'      # Default ordering
_rec_name = 'display_field'     # Display name field
_parent_name = 'parent_id'      # Parent field for hierarchy
_parent_store = True            # Enable materialized path
_auto = True                    # Auto-create table
_log_access = True              # Log create/write info
_transient = False              # Transient model flag
_check_company_auto = True      # Auto company validation
_allow_sudo_commands = True     # Allow sudo on X2many
```

### Environment Access

```python
self.env                    # Environment
self.env.user              # Current user
self.env.company           # Current company
self.env.companies         # Allowed companies
self.env.uid               # Current user ID
self.env.context           # Context dictionary
self.env.lang              # Current language
self.env.su                # Is superuser mode
self.env.cr                # Database cursor

self.sudo()                # Superuser mode
self.with_user(user)       # Different user
self.with_company(company) # Different company
self.with_context(key=val) # Add context
self.with_env(new_env)     # Different environment
```

### Commands for X2many

```python
from odoo import Command

# One2many / Many2many write commands
(0, 0, {'field': value})           # Create new
(1, id, {'field': value})          # Update existing
(2, id, 0)                         # Delete (unlink)
(3, id, 0)                         # Unlink (M2M only)
(4, id, 0)                         # Link existing (M2M)
(5, 0, 0)                          # Unlink all (M2M)
(6, 0, [ids])                      # Replace all (M2M)

# Using Command helper (recommended)
Command.create({'field': value})   # Create
Command.update(id, {'field': val}) # Update
Command.delete(id)                 # Delete
Command.unlink(id)                 # Unlink (M2M)
Command.link(id)                   # Link (M2M)
Command.clear()                    # Clear all (M2M)
Command.set([ids])                 # Set list (M2M)
```

---

*Document Version: 2.0*
*Last Updated: December 2025*
*Odoo Versions Covered: 14.0 - 19.0 (preview)*
