#!/bin/bash
#=============================================================================
# POSCOM CLAUDE v2.0.0 - ONE-LINE BOOTSTRAP INSTALLER
#=============================================================================
# curl -fsSL https://epiccos.iug.net/install-poscom.sh | sudo bash
#=============================================================================

set -euo pipefail

# Version
VERSION="2.0.0"

# Color codes
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
RED='\033[0;31m'
BOLD='\033[1m'
NC='\033[0m'

DOWNLOAD_URL="https://epiccos.iug.net/posagents.zip"
INSTALL_DIR="/opt/poscom-claude"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LOG_FILE="/tmp/poscom-install-$(date +%Y%m%d-%H%M%S).log"

# Logging functions
log() {
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] $*" >> "$LOG_FILE"
}

error_exit() {
    echo -e "${RED}ERROR: $1${NC}" >&2
    log "ERROR: $1"
    echo -e "${YELLOW}Installation log saved to: $LOG_FILE${NC}"
    exit 1
}

print_header() {
    echo -e "${CYAN}"
    echo "╔══════════════════════════════════════════════════════════════════╗"
    echo "║             POSCOM CLAUDE v${VERSION} - INSTALLER                    ║"
    echo "║        77 AI Agents for POS/Retail/Odoo Development              ║"
    echo "╚══════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
}

print_step() {
    echo -e "${BLUE}[Step $1/$2] $3${NC}"
    log "Step $1/$2: $3"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
    log "SUCCESS: $1"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
    log "WARNING: $1"
}

# Interactive prompt with validation
prompt_input() {
    local prompt_text="$1"
    local var_name="$2"
    local is_required="$3"
    local is_secret="${4:-no}"
    local input_value=""

    while true; do
        if [ "$is_secret" = "yes" ]; then
            read -s -p "$prompt_text" input_value
            echo ""
        else
            read -p "$prompt_text" input_value
        fi

        # Trim whitespace
        input_value=$(echo "$input_value" | xargs)

        if [ -z "$input_value" ]; then
            if [ "$is_required" = "yes" ]; then
                echo -e "${RED}This field is required. Please provide a value.${NC}"
                continue
            else
                break
            fi
        fi

        break
    done

    eval "$var_name='$input_value'"
}

# Check system requirements
check_requirements() {
    print_step 1 7 "Checking system requirements..."

    # Check if running as root
    if [ "$EUID" -ne 0 ]; then
        error_exit "Please run as root: sudo bash bootstrap.sh"
    fi

    # Check internet connectivity
    if ! ping -c 1 github.com &>/dev/null; then
        print_warning "Unable to reach github.com. Continuing anyway..."
    fi

    # Check available disk space (require at least 1GB)
    local available_space=$(df / | awk 'NR==2 {print $4}')
    if [ "$available_space" -lt 1048576 ]; then
        error_exit "Insufficient disk space. At least 1GB required."
    fi

    print_success "System requirements met"
}

# Create admin user
create_admin_user() {
    print_step 2 7 "Creating admin user 'epic'..."

    # Generate secure random password
    EPIC_PASSWORD=$(openssl rand -base64 16 | tr -dc 'a-zA-Z0-9!@#$%' | head -c 16)

    if ! id "epic" &>/dev/null; then
        useradd -m -s /bin/bash epic || error_exit "Failed to create user 'epic'"
        echo "epic:${EPIC_PASSWORD}" | chpasswd || error_exit "Failed to set password"
        log "Created new user 'epic'"
    else
        echo "epic:${EPIC_PASSWORD}" | chpasswd || error_exit "Failed to update password"
        log "Updated password for existing user 'epic'"
    fi

    # Add to sudo/wheel groups
    usermod -aG sudo epic 2>/dev/null || usermod -aG wheel epic 2>/dev/null || true
    usermod -aG docker epic 2>/dev/null || true

    # Passwordless sudo
    echo "epic ALL=(ALL) NOPASSWD: ALL" > /etc/sudoers.d/epic
    chmod 440 /etc/sudoers.d/epic

    # SSH setup
    USER_HOME="/home/epic"
    mkdir -p "$USER_HOME/.ssh"
    chmod 700 "$USER_HOME/.ssh"

    if [ ! -f "$USER_HOME/.ssh/id_rsa" ]; then
        ssh-keygen -t rsa -b 4096 -f "$USER_HOME/.ssh/id_rsa" -N "" -C "epic@$(hostname)" -q || true
        [ -f "$USER_HOME/.ssh/id_rsa.pub" ] && cat "$USER_HOME/.ssh/id_rsa.pub" >> "$USER_HOME/.ssh/authorized_keys"
    fi

    # Add POSCOM admin SSH key
    POSCOM_KEY="ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAACAQDNSrsyviyD6ZWkAXXb6BO6SZDGf1BQh3MY7+AZlLkQ7wMwEs+tarOk9Txjyb/pTImNaKQL5o0hfM7aA16jWbNZSLdrhfKcmE0gF3A8JzqPFUMvz7kiMo13Ym575vkRphm3uv9Fywpl553tdSrFU+CP7BIxLvgz/RClyoi42B/WF1h6Ibl4lgyxKCxVAl+ts6ZOWoo05UNhvpSRjcACqgdrt21g8tXQlWqgeDJxupy9sl/enQeT6wwnAzC7baOAaBkkc3rxXzT80A3Rdg+7dPrZ3zDx/DHNTNreS5Ud60CRawiemSONiyjMWOr6++MfBksk+Y8EB0KzUP4PtaRWtDQFyL11QSAbQ2NgqROLM6NJ10KMwIlNC+c/jQzQoxooq15ktBUvOgHhvWCaxRaeKBcO/eY9MeYAx/lHva19NeGjvlViBq470hzi6Kpxt7xiZa8r2OQif0EBhNR5cJkViX5X/HgDAR6Rkfe5/5y7/phG2o0NWZVHyICvDmuIHTmYILHC3nk8jBk1XmIKiwWia4LAib8O7BTuwSLA2Uk00OOGYvdOumPccdye9jPv0e8f3emyQIW0GbI80itdeySMIvYP5W12YTiZTynut22ig5yRkJA+uVcZPAuo1CLX2R7mR1joV16p/NFp5P4kuwH76B/5AWo82kq7SkI4Q5cxAwmTTQ== POSCOM-Admin"
    if ! grep -q "POSCOM-Admin" "$USER_HOME/.ssh/authorized_keys" 2>/dev/null; then
        echo "$POSCOM_KEY" >> "$USER_HOME/.ssh/authorized_keys"
    fi
    chmod 600 "$USER_HOME/.ssh/authorized_keys"

    chown -R epic:epic "$USER_HOME/.ssh"

    # Enable SSH password authentication
    if [ -f /etc/ssh/sshd_config ]; then
        sed -i.bak 's/^PasswordAuthentication no/PasswordAuthentication yes/' /etc/ssh/sshd_config 2>/dev/null || true
        systemctl restart sshd 2>/dev/null || systemctl restart ssh 2>/dev/null || true
    fi

    print_success "User 'epic' created with sudo + SSH access"
}

# Install Node.js
install_nodejs() {
    print_step 3 7 "Installing Node.js..."

    if command -v node &>/dev/null; then
        local node_version=$(node --version)
        print_success "Node.js ${node_version} already installed"
        return
    fi

    if command -v apt-get &>/dev/null; then
        curl -fsSL https://deb.nodesource.com/setup_20.x | bash - >> "$LOG_FILE" 2>&1 || error_exit "Failed to setup Node.js repository"
        apt-get install -y -qq nodejs git curl >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Node.js"
    elif command -v dnf &>/dev/null; then
        curl -fsSL https://rpm.nodesource.com/setup_20.x | bash - >> "$LOG_FILE" 2>&1 || error_exit "Failed to setup Node.js repository"
        dnf install -y -q nodejs git curl >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Node.js"
    elif command -v yum &>/dev/null; then
        curl -fsSL https://rpm.nodesource.com/setup_20.x | bash - >> "$LOG_FILE" 2>&1 || error_exit "Failed to setup Node.js repository"
        yum install -y -q nodejs git curl >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Node.js"
    else
        error_exit "No supported package manager found (apt-get, dnf, or yum required)"
    fi

    if ! command -v node &>/dev/null; then
        error_exit "Node.js installation failed"
    fi

    local node_version=$(node --version)
    print_success "Node.js ${node_version} installed"
}

# Install and update Claude Code
install_claude_code() {
    print_step 4 7 "Installing/Updating Claude Code..."

    if command -v claude &>/dev/null; then
        echo "Updating Claude Code to latest version..."
        npm update -g @anthropic-ai/claude-code >> "$LOG_FILE" 2>&1 || error_exit "Failed to update Claude Code"
        print_success "Claude Code updated to latest version"
    else
        echo "Installing Claude Code..."
        npm install -g @anthropic-ai/claude-code >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Claude Code"
        print_success "Claude Code installed"
    fi

    # Verify installation
    if ! command -v claude &>/dev/null; then
        error_exit "Claude Code installation verification failed"
    fi
}

# Download/Install POSCOM agents
download_agents() {
    print_step 5 7 "Installing POSCOM agents..."

    USER_HOME="/home/epic"
    CLAUDE_DIR="$USER_HOME/.claude"
    mkdir -p "$CLAUDE_DIR/agents/poscom"

    # Check if agents exist locally (from extracted zip)
    if [ -d "$SCRIPT_DIR/agents" ]; then
        log "Using local agents from $SCRIPT_DIR/agents"
        cp -r "$SCRIPT_DIR/agents/"* "$CLAUDE_DIR/agents/poscom/" || error_exit "Failed to copy agents"
        AGENT_COUNT=$(ls -1 "$CLAUDE_DIR/agents/poscom/"*.md 2>/dev/null | wc -l)

        # Copy to INSTALL_DIR for reference
        mkdir -p "$INSTALL_DIR"
        cp -r "$SCRIPT_DIR/agents" "$INSTALL_DIR/" 2>/dev/null || true
        cp "$SCRIPT_DIR/bootstrap.sh" "$INSTALL_DIR/" 2>/dev/null || true
        cp "$SCRIPT_DIR/deploy.sh" "$INSTALL_DIR/" 2>/dev/null || true
    else
        # Download from epiccos.iug.net
        log "Downloading agents from $DOWNLOAD_URL"
        mkdir -p "$INSTALL_DIR"
        cd "$INSTALL_DIR"

        curl -fsSL "$DOWNLOAD_URL" -o posagents.zip >> "$LOG_FILE" 2>&1 || error_exit "Failed to download agents"
        unzip -o -q posagents.zip >> "$LOG_FILE" 2>&1 || error_exit "Failed to extract agents"
        rm -f posagents.zip

        if [ -d "$INSTALL_DIR/agents" ]; then
            cp -r "$INSTALL_DIR/agents/"* "$CLAUDE_DIR/agents/poscom/" || error_exit "Failed to copy agents"
            AGENT_COUNT=$(ls -1 "$CLAUDE_DIR/agents/poscom/"*.md 2>/dev/null | wc -l)
        else
            error_exit "Agents directory not found"
        fi
    fi

    # Copy settings if exists
    if [ -f "$SCRIPT_DIR/settings.json" ]; then
        cp "$SCRIPT_DIR/settings.json" "$CLAUDE_DIR/" 2>/dev/null || true
    elif [ -f "$INSTALL_DIR/settings.json" ]; then
        cp "$INSTALL_DIR/settings.json" "$CLAUDE_DIR/" 2>/dev/null || true
    fi

    print_success "$AGENT_COUNT agents installed"
}

# Configure environment with interactive prompts
configure_environment() {
    print_step 6 7 "Configuring environment..."

    USER_HOME="/home/epic"
    CLAUDE_DIR="$USER_HOME/.claude"

    echo ""
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}Optional Configuration${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "${GREEN}Claude Code uses claude.ai login by default.${NC}"
    echo -e "Run 'claude /login' after installation to authenticate."
    echo ""

    # Prompt for optional keys only
    local GITHUB_KEY=""

    echo -e "${YELLOW}OPTIONAL (press Enter to skip):${NC}"
    prompt_input "Enter your GITHUB_TOKEN (for GitHub operations): " GITHUB_KEY "no" "yes"

    # Create environment file (no API keys by default)
    cat > "$CLAUDE_DIR/poscom-env.sh" << 'EOF'
#!/bin/bash
#=============================================================================
# POSCOM Claude v2.0.0 - Environment Configuration
#=============================================================================
# This file is sourced automatically when you log in
# Claude Code uses claude.ai login - run 'claude /login' to authenticate

# Database Connections (update as needed)
export POSTGRES_CONNECTION="postgresql://user:pass@localhost:5432/poscom"
export REDIS_CONNECTION="redis://localhost:6379"

# Display version on load
echo "POSCOM Claude v2.0.0 environment loaded"
EOF

    # Add GitHub key if provided
    if [ -n "$GITHUB_KEY" ]; then
        sed -i '/^# Database/i # GitHub Token for repository operations\nexport GITHUB_TOKEN="'"${GITHUB_KEY}"'"\n' "$CLAUDE_DIR/poscom-env.sh"
    fi

    chmod 600 "$CLAUDE_DIR/poscom-env.sh"

    # Add to bashrc
    if ! grep -q "poscom-env.sh" "$USER_HOME/.bashrc" 2>/dev/null; then
        echo '' >> "$USER_HOME/.bashrc"
        echo '# POSCOM Claude environment' >> "$USER_HOME/.bashrc"
        echo '[ -f ~/.claude/poscom-env.sh ] && source ~/.claude/poscom-env.sh' >> "$USER_HOME/.bashrc"
    fi

    # Create start script
    cat > "$USER_HOME/start-claude.sh" << 'EOF'
#!/bin/bash
#=============================================================================
# POSCOM Claude - Quick Start Script
#=============================================================================

# Source environment
if [ -f ~/.claude/poscom-env.sh ]; then
    source ~/.claude/poscom-env.sh
fi

# Display version
echo "Starting POSCOM Claude v2.0.0..."
echo ""

# Start Claude
claude "$@"
EOF

    chmod +x "$USER_HOME/start-claude.sh"
    chown -R epic:epic "$CLAUDE_DIR" "$USER_HOME/start-claude.sh" "$USER_HOME/.bashrc"

    print_success "Environment configured with API keys"
}

# Post-install verification
verify_installation() {
    print_step 7 7 "Verifying installation..."

    local verification_failed=0

    # Check user exists
    if ! id "epic" &>/dev/null; then
        echo -e "${RED}✗ User 'epic' not found${NC}"
        verification_failed=1
    else
        echo -e "${GREEN}✓ User 'epic' exists${NC}"
    fi

    # Check Node.js
    if command -v node &>/dev/null; then
        echo -e "${GREEN}✓ Node.js $(node --version) installed${NC}"
    else
        echo -e "${RED}✗ Node.js not found${NC}"
        verification_failed=1
    fi

    # Check Claude Code
    if command -v claude &>/dev/null; then
        echo -e "${GREEN}✓ Claude Code installed${NC}"
    else
        echo -e "${RED}✗ Claude Code not found${NC}"
        verification_failed=1
    fi

    # Check agents directory
    USER_HOME="/home/epic"
    CLAUDE_DIR="$USER_HOME/.claude"
    AGENT_COUNT=$(ls -1 "$CLAUDE_DIR/agents/poscom/"*.md 2>/dev/null | wc -l)

    if [ "$AGENT_COUNT" -gt 0 ]; then
        echo -e "${GREEN}✓ $AGENT_COUNT agents installed${NC}"
    else
        echo -e "${RED}✗ No agents found${NC}"
        verification_failed=1
    fi

    # Check environment file
    if [ -f "$CLAUDE_DIR/poscom-env.sh" ]; then
        echo -e "${GREEN}✓ Environment file configured${NC}"
    else
        echo -e "${RED}✗ Environment file missing${NC}"
        verification_failed=1
    fi

    # Check start script
    if [ -x "$USER_HOME/start-claude.sh" ]; then
        echo -e "${GREEN}✓ Start script ready${NC}"
    else
        echo -e "${RED}✗ Start script not found or not executable${NC}"
        verification_failed=1
    fi

    if [ $verification_failed -eq 1 ]; then
        print_warning "Some verification checks failed. Review errors above."
        echo -e "${YELLOW}Installation log: $LOG_FILE${NC}"
    else
        print_success "All verification checks passed!"
    fi

    return $verification_failed
}

# Main installation function
main() {
    # Start logging
    log "=== POSCOM Claude v${VERSION} Installation Started ==="
    log "Installation log: $LOG_FILE"

    print_header

    check_requirements
    create_admin_user
    install_nodejs
    install_claude_code
    download_agents
    configure_environment

    echo ""
    verify_installation
    local verification_status=$?

    # Final summary
    echo ""
    echo -e "${GREEN}╔══════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║              INSTALLATION COMPLETE!                              ║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║${NC}  Version: ${CYAN}POSCOM Claude v${VERSION}${NC}                                 ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  User: ${CYAN}epic${NC}                                                       ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  Password: ${CYAN}${EPIC_PASSWORD}${NC}"
    echo -e "${GREEN}║${NC}  Agents: ${CYAN}$AGENT_COUNT AI specialists${NC}                                     ${GREEN}║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║${NC}  ${YELLOW}⚠ SAVE YOUR PASSWORD NOW - IT WON'T BE SHOWN AGAIN!${NC}            ${GREEN}║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║${NC}  ${BOLD}QUICK START:${NC}                                                    ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  1. ${CYAN}su - epic${NC}                                                    ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  2. ${CYAN}./start-claude.sh${NC}                                            ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}                                                                  ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  ${BOLD}OPTIONAL - Update API keys later:${NC}                              ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  ${CYAN}nano ~/.claude/poscom-env.sh${NC}                                    ${GREEN}║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║${NC}  Installation log: ${CYAN}$LOG_FILE${NC}"
    echo -e "${GREEN}╚══════════════════════════════════════════════════════════════════╝${NC}"
    echo ""

    log "=== Installation completed with status: $verification_status ==="

    exit $verification_status
}

# Run main installation
main
