#!/bin/bash
#===============================================================================
# POSCOM Claude v2.0.0 - One-Click Deployment
#===============================================================================
# Run: sudo ./deploy.sh
#===============================================================================

set -euo pipefail

# Version
VERSION="2.0.0"

# Get script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Color codes
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
RED='\033[0;31m'
BOLD='\033[1m'
NC='\033[0m'

LOG_FILE="/tmp/poscom-deploy-$(date +%Y%m%d-%H%M%S).log"

# Logging functions
log() {
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] $*" >> "$LOG_FILE"
}

error_exit() {
    echo -e "${RED}ERROR: $1${NC}" >&2
    log "ERROR: $1"
    echo -e "${YELLOW}Deployment log saved to: $LOG_FILE${NC}"
    exit 1
}

print_header() {
    echo -e "${CYAN}"
    echo "╔══════════════════════════════════════════════════════════════════╗"
    echo "║           POSCOM CLAUDE v${VERSION} - DEPLOYMENT                     ║"
    echo "║        77 AI Agents for POS/Retail/Odoo Development              ║"
    echo "╚══════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
}

print_step() {
    echo -e "${BLUE}[Step $1/$2] $3${NC}"
    log "Step $1/$2: $3"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
    log "SUCCESS: $1"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
    log "WARNING: $1"
}

# Interactive prompt with validation
prompt_input() {
    local prompt_text="$1"
    local var_name="$2"
    local is_required="$3"
    local is_secret="${4:-no}"
    local input_value=""

    while true; do
        if [ "$is_secret" = "yes" ]; then
            read -s -p "$prompt_text" input_value
            echo ""
        else
            read -p "$prompt_text" input_value
        fi

        # Trim whitespace
        input_value=$(echo "$input_value" | xargs)

        if [ -z "$input_value" ]; then
            if [ "$is_required" = "yes" ]; then
                echo -e "${RED}This field is required. Please provide a value.${NC}"
                continue
            else
                break
            fi
        fi

        break
    done

    eval "$var_name='$input_value'"
}

# Create admin user (if running as root)
create_admin_user() {
    if [ "$EUID" -ne 0 ]; then
        return 0
    fi

    print_step 1 6 "Creating admin user 'epic'..."

    # Generate secure random password
    EPIC_PASSWORD=$(openssl rand -base64 16 | tr -dc 'a-zA-Z0-9!@#$%' | head -c 16)

    if ! id "epic" &>/dev/null; then
        useradd -m -s /bin/bash epic || error_exit "Failed to create user 'epic'"
        echo "epic:${EPIC_PASSWORD}" | chpasswd || error_exit "Failed to set password"
        log "Created new user 'epic'"
        print_success "User 'epic' created"
        echo -e "${YELLOW}Password: ${EPIC_PASSWORD}${NC}"
        echo -e "${YELLOW}SAVE THIS PASSWORD - IT WON'T BE SHOWN AGAIN!${NC}"
    else
        echo "epic:${EPIC_PASSWORD}" | chpasswd || error_exit "Failed to reset password"
        log "Reset password for existing user 'epic'"
        print_success "User 'epic' password reset"
        echo -e "${YELLOW}New password: ${EPIC_PASSWORD}${NC}"
        echo -e "${YELLOW}SAVE THIS PASSWORD - IT WON'T BE SHOWN AGAIN!${NC}"
    fi

    # Add to sudo/wheel groups
    usermod -aG sudo epic 2>/dev/null || usermod -aG wheel epic 2>/dev/null || true
    usermod -aG docker epic 2>/dev/null || true

    # Passwordless sudo
    echo "epic ALL=(ALL) NOPASSWD: ALL" > /etc/sudoers.d/epic
    chmod 440 /etc/sudoers.d/epic

    # Setup SSH
    USER_HOME="/home/epic"
    mkdir -p "$USER_HOME/.ssh"
    chmod 700 "$USER_HOME/.ssh"

    if [ ! -f "$USER_HOME/.ssh/id_rsa" ]; then
        ssh-keygen -t rsa -b 4096 -f "$USER_HOME/.ssh/id_rsa" -N "" -C "epic@$(hostname)" -q || true
        [ -f "$USER_HOME/.ssh/id_rsa.pub" ] && cat "$USER_HOME/.ssh/id_rsa.pub" >> "$USER_HOME/.ssh/authorized_keys"
        chmod 600 "$USER_HOME/.ssh/authorized_keys" 2>/dev/null || true
    fi

    chown -R epic:epic "$USER_HOME/.ssh"

    # Enable SSH password authentication
    if [ -f /etc/ssh/sshd_config ]; then
        sed -i.bak 's/^PasswordAuthentication no/PasswordAuthentication yes/' /etc/ssh/sshd_config 2>/dev/null || true
        systemctl restart sshd 2>/dev/null || systemctl restart ssh 2>/dev/null || true
    fi

    print_success "Admin user configured with SSH access"
}

# Install Node.js
install_nodejs() {
    local step_num=2
    [ "$EUID" -ne 0 ] && step_num=1

    print_step $step_num 6 "Checking Node.js installation..."

    if command -v node &>/dev/null; then
        local node_version=$(node --version)
        print_success "Node.js ${node_version} already installed"
        return
    fi

    if [ "$EUID" -ne 0 ]; then
        error_exit "Node.js not found. Please run as root to install dependencies."
    fi

    echo "Installing Node.js..."
    if command -v apt-get &> /dev/null; then
        curl -fsSL https://deb.nodesource.com/setup_20.x | bash - >> "$LOG_FILE" 2>&1 || error_exit "Failed to setup Node.js repository"
        apt-get install -y -qq nodejs git curl >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Node.js"
    elif command -v dnf &> /dev/null; then
        curl -fsSL https://rpm.nodesource.com/setup_20.x | bash - >> "$LOG_FILE" 2>&1 || error_exit "Failed to setup Node.js repository"
        dnf install -y -q nodejs git curl >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Node.js"
    elif command -v yum &> /dev/null; then
        curl -fsSL https://rpm.nodesource.com/setup_20.x | bash - >> "$LOG_FILE" 2>&1 || error_exit "Failed to setup Node.js repository"
        yum install -y -q nodejs git curl >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Node.js"
    else
        error_exit "No supported package manager found (apt-get, dnf, or yum required)"
    fi

    if ! command -v node &>/dev/null; then
        error_exit "Node.js installation failed"
    fi

    local node_version=$(node --version)
    print_success "Node.js ${node_version} installed"
}

# Install and update Claude Code
install_claude_code() {
    local step_num=3
    [ "$EUID" -ne 0 ] && step_num=2

    print_step $step_num 6 "Installing/Updating Claude Code..."

    if command -v claude &>/dev/null; then
        echo "Updating Claude Code to latest version..."
        npm update -g @anthropic-ai/claude-code >> "$LOG_FILE" 2>&1 || error_exit "Failed to update Claude Code"
        print_success "Claude Code updated to latest version"
    else
        echo "Installing Claude Code..."
        npm install -g @anthropic-ai/claude-code >> "$LOG_FILE" 2>&1 || error_exit "Failed to install Claude Code"
        print_success "Claude Code installed"
    fi

    # Verify installation
    if ! command -v claude &>/dev/null; then
        error_exit "Claude Code installation verification failed"
    fi
}

# Deploy agents and configuration
deploy_agents() {
    local step_num=4
    [ "$EUID" -ne 0 ] && step_num=3

    print_step $step_num 6 "Deploying POSCOM agents..."

    # Create Claude directories
    mkdir -p ~/.claude/agents/poscom

    # Copy agents - check local first, download if needed
    if [ -d "$SCRIPT_DIR/agents" ]; then
        log "Using local agents from $SCRIPT_DIR/agents"
        cp -r "$SCRIPT_DIR/agents/"* ~/.claude/agents/poscom/ || error_exit "Failed to copy agents"
    else
        # Download from epiccos.iug.net
        log "Downloading agents from https://epiccos.iug.net/posagents.zip"
        local TEMP_DIR="/tmp/poscom-download-$$"
        mkdir -p "$TEMP_DIR"
        cd "$TEMP_DIR"
        curl -fsSL "https://epiccos.iug.net/posagents.zip" -o posagents.zip >> "$LOG_FILE" 2>&1 || error_exit "Failed to download agents"
        unzip -o -q posagents.zip >> "$LOG_FILE" 2>&1 || error_exit "Failed to extract agents"
        cp -r "$TEMP_DIR/agents/"* ~/.claude/agents/poscom/ || error_exit "Failed to copy agents"
        rm -rf "$TEMP_DIR"
        cd - > /dev/null
    fi

    AGENT_COUNT=$(ls -1 ~/.claude/agents/poscom/*.md 2>/dev/null | wc -l)

    # Copy settings
    if [ -f "$SCRIPT_DIR/settings.json" ]; then
        cp "$SCRIPT_DIR/settings.json" ~/.claude/settings.json
        chmod 600 ~/.claude/settings.json
        log "Settings file copied"
    fi

    print_success "$AGENT_COUNT agents deployed"
}

# Configure environment with interactive prompts
configure_environment() {
    local step_num=5
    [ "$EUID" -ne 0 ] && step_num=4

    print_step $step_num 6 "Configuring environment..."

    echo ""
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}API Key Configuration${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""

    # Prompt for API keys
    local ANTHROPIC_KEY=""
    local GITHUB_KEY=""
    local GOOGLE_KEY=""

    echo -e "${YELLOW}REQUIRED:${NC}"
    prompt_input "Enter your ANTHROPIC_API_KEY: " ANTHROPIC_KEY "yes" "yes"

    echo ""
    echo -e "${YELLOW}OPTIONAL (press Enter to skip):${NC}"
    prompt_input "Enter your GITHUB_TOKEN (for GitHub operations): " GITHUB_KEY "no" "yes"
    prompt_input "Enter your GOOGLE_AI_API_KEY (for Gemini features): " GOOGLE_KEY "no" "yes"

    # Create environment file
    cat > ~/.claude/poscom-env.sh << EOF
#!/bin/bash
#=============================================================================
# POSCOM Claude v${VERSION} - Environment Configuration
#=============================================================================
# This file is sourced automatically when you log in
# You can edit this file later to update your API keys

# Required: Anthropic API Key for Claude
export ANTHROPIC_API_KEY="${ANTHROPIC_KEY}"

EOF

    # Add optional keys if provided
    if [ -n "$GITHUB_KEY" ]; then
        cat >> ~/.claude/poscom-env.sh << EOF
# Optional: GitHub Token for repository operations
export GITHUB_TOKEN="${GITHUB_KEY}"

EOF
    fi

    if [ -n "$GOOGLE_KEY" ]; then
        cat >> ~/.claude/poscom-env.sh << EOF
# Optional: Google AI API Key for Gemini features
export GOOGLE_AI_API_KEY="${GOOGLE_KEY}"

EOF
    fi

    # Add database connections (user can update later)
    cat >> ~/.claude/poscom-env.sh << 'EOF'
# Database Connections (update as needed)
export POSTGRES_CONNECTION="postgresql://user:pass@localhost:5432/poscom"
export REDIS_CONNECTION="redis://localhost:6379"

# Display version on load
echo "POSCOM Claude v2.0.0 environment loaded"
EOF

    chmod 600 ~/.claude/poscom-env.sh

    # Add to shell profile
    PROFILE="${HOME}/.bashrc"
    [ -f "${HOME}/.zshrc" ] && PROFILE="${HOME}/.zshrc"

    if ! grep -q "poscom-env.sh" "$PROFILE" 2>/dev/null; then
        echo '' >> "$PROFILE"
        echo '# POSCOM Claude environment' >> "$PROFILE"
        echo '[ -f ~/.claude/poscom-env.sh ] && source ~/.claude/poscom-env.sh' >> "$PROFILE"
        log "Added environment loading to $PROFILE"
    fi

    # Create start script
    cat > ~/start-claude.sh << 'EOF'
#!/bin/bash
#=============================================================================
# POSCOM Claude - Quick Start Script
#=============================================================================

# Source environment
if [ -f ~/.claude/poscom-env.sh ]; then
    source ~/.claude/poscom-env.sh
else
    echo "Error: Environment file not found at ~/.claude/poscom-env.sh"
    exit 1
fi

# Verify API key is set
if [ -z "$ANTHROPIC_API_KEY" ] || [ "$ANTHROPIC_API_KEY" = "your-anthropic-api-key" ]; then
    echo "Error: ANTHROPIC_API_KEY not configured"
    echo "Please edit ~/.claude/poscom-env.sh and add your API key"
    exit 1
fi

# Display version
echo "Starting POSCOM Claude v2.0.0..."
echo ""

# Start Claude
claude "$@"
EOF

    chmod +x ~/start-claude.sh
    log "Created start script"

    print_success "Environment configured with API keys"
}

# Post-deployment verification
verify_deployment() {
    local step_num=6
    [ "$EUID" -ne 0 ] && step_num=5

    print_step $step_num 6 "Verifying deployment..."

    local verification_failed=0

    # Check Node.js
    if command -v node &>/dev/null; then
        echo -e "${GREEN}✓ Node.js $(node --version) installed${NC}"
    else
        echo -e "${RED}✗ Node.js not found${NC}"
        verification_failed=1
    fi

    # Check Claude Code
    if command -v claude &>/dev/null; then
        echo -e "${GREEN}✓ Claude Code installed${NC}"
    else
        echo -e "${RED}✗ Claude Code not found${NC}"
        verification_failed=1
    fi

    # Check agents directory
    AGENT_COUNT=$(ls -1 ~/.claude/agents/poscom/*.md 2>/dev/null | wc -l)
    if [ "$AGENT_COUNT" -gt 0 ]; then
        echo -e "${GREEN}✓ $AGENT_COUNT agents deployed${NC}"
    else
        echo -e "${RED}✗ No agents found${NC}"
        verification_failed=1
    fi

    # Check environment file
    if [ -f ~/.claude/poscom-env.sh ]; then
        echo -e "${GREEN}✓ Environment file configured${NC}"
    else
        echo -e "${RED}✗ Environment file missing${NC}"
        verification_failed=1
    fi

    # Check start script
    if [ -x ~/start-claude.sh ]; then
        echo -e "${GREEN}✓ Start script ready${NC}"
    else
        echo -e "${RED}✗ Start script not found or not executable${NC}"
        verification_failed=1
    fi

    if [ $verification_failed -eq 1 ]; then
        print_warning "Some verification checks failed. Review errors above."
        echo -e "${YELLOW}Deployment log: $LOG_FILE${NC}"
    else
        print_success "All verification checks passed!"
    fi

    return $verification_failed
}

# Main deployment function
main() {
    # Start logging
    log "=== POSCOM Claude v${VERSION} Deployment Started ==="
    log "Deployment log: $LOG_FILE"
    log "Running as user: $(whoami)"
    log "Running as root: $([[ $EUID -eq 0 ]] && echo 'yes' || echo 'no')"

    print_header

    # Execute deployment steps
    create_admin_user  # Only runs if root
    install_nodejs
    install_claude_code
    deploy_agents
    configure_environment

    echo ""
    verify_deployment
    local verification_status=$?

    # Final summary
    echo ""
    echo -e "${GREEN}╔══════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║              DEPLOYMENT COMPLETE!                                ║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║${NC}  Version: ${CYAN}POSCOM Claude v${VERSION}${NC}                                 ${GREEN}║${NC}"

    if [ -n "${EPIC_PASSWORD:-}" ]; then
        echo -e "${GREEN}║${NC}  User: ${CYAN}epic${NC}                                                       ${GREEN}║${NC}"
        echo -e "${GREEN}║${NC}  Password: ${CYAN}${EPIC_PASSWORD}${NC}"
    else
        echo -e "${GREEN}║${NC}  User: ${CYAN}$(whoami)${NC}                                                   ${GREEN}║${NC}"
    fi

    echo -e "${GREEN}║${NC}  Agents: ${CYAN}$AGENT_COUNT AI specialists${NC}                                     ${GREEN}║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"

    if [ -n "${EPIC_PASSWORD:-}" ]; then
        echo -e "${GREEN}║${NC}  ${YELLOW}⚠ SAVE YOUR PASSWORD NOW - IT WON'T BE SHOWN AGAIN!${NC}            ${GREEN}║${NC}"
        echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    fi

    echo -e "${GREEN}║${NC}  ${BOLD}QUICK START:${NC}                                                    ${GREEN}║${NC}"

    if [ -n "${EPIC_PASSWORD:-}" ]; then
        echo -e "${GREEN}║${NC}  1. ${CYAN}su - epic${NC}                                                    ${GREEN}║${NC}"
        echo -e "${GREEN}║${NC}  2. ${CYAN}./start-claude.sh${NC}                                            ${GREEN}║${NC}"
    else
        echo -e "${GREEN}║${NC}  1. ${CYAN}source ~/.claude/poscom-env.sh${NC}                               ${GREEN}║${NC}"
        echo -e "${GREEN}║${NC}  2. ${CYAN}./start-claude.sh${NC}                                            ${GREEN}║${NC}"
    fi

    echo -e "${GREEN}║${NC}                                                                  ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  ${BOLD}OPTIONAL - Update API keys later:${NC}                              ${GREEN}║${NC}"
    echo -e "${GREEN}║${NC}  ${CYAN}nano ~/.claude/poscom-env.sh${NC}                                    ${GREEN}║${NC}"
    echo -e "${GREEN}╠══════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${GREEN}║${NC}  Deployment log: ${CYAN}$LOG_FILE${NC}"
    echo -e "${GREEN}╚══════════════════════════════════════════════════════════════════╝${NC}"
    echo ""

    log "=== Deployment completed with status: $verification_status ==="

    exit $verification_status
}

# Run main deployment
main
